<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once __DIR__ . "/config.php";

/*
|--------------------------------------------------------------------------
|  BRUTE-FORCE PROTECTION
|--------------------------------------------------------------------------
|  - Limits failed login attempts
|  - Tracks by IP address
|  - Blocks repeated failures for 10 minutes
|--------------------------------------------------------------------------
*/

function is_rate_limited() {
    $ip = $_SERVER['REMOTE_ADDR'];
    $key = "login_attempts_" . $ip;

    if (!isset($_SESSION[$key])) {
        $_SESSION[$key] = [
            "attempts" => 0,
            "last_attempt" => time()
        ];
    }

    $data = $_SESSION[$key];

    // If blocked
    if ($data["attempts"] >= 5 && (time() - $data["last_attempt"]) < 600) {
        return true; // BLOCK
    }

    return false;
}

function record_failed_attempt() {
    $ip = $_SERVER['REMOTE_ADDR'];
    $key = "login_attempts_" . $ip;

    if (!isset($_SESSION[$key])) {
        $_SESSION[$key] = ["attempts" => 0, "last_attempt" => time()];
    }

    $_SESSION[$key]["attempts"]++;
    $_SESSION[$key]["last_attempt"] = time();
}

function reset_attempts() {
    $ip = $_SERVER['REMOTE_ADDR'];
    $key = "login_attempts_" . $ip;
    unset($_SESSION[$key]);
}

/*
|--------------------------------------------------------------------------
| LOGIN HANDLER (EMAIL ONLY – MOST SECURE)
|--------------------------------------------------------------------------
*/

function login_admin($email, $password) {
    global $pdo;

    // Email validation
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        return false;
    }

    // Check user by email
    $stmt = $pdo->prepare("SELECT * FROM admin_users WHERE email = ? LIMIT 1");
    $stmt->execute([$email]);
    $user = $stmt->fetch();

    // Always run password_verify to prevent timing attacks
    if (!$user || !password_verify($password, $user['password'])) {
        record_failed_attempt();
        return false;
    }

    // At this point credentials are valid
    reset_attempts();

    // Session Security
    session_regenerate_id(true);

    $_SESSION['admin_id'] = $user['id'];
    $_SESSION['admin_email'] = $user['email'];
    $_SESSION['admin_username'] = $user['username'];
    $_SESSION['admin_logged'] = true;
    $_SESSION['ip_address'] = $_SERVER['REMOTE_ADDR'];
    $_SESSION['user_agent'] = $_SERVER['HTTP_USER_AGENT'];

    return true;
}

/*
|--------------------------------------------------------------------------
| CHECK IF USER IS LOGGED IN
|--------------------------------------------------------------------------
*/

function is_logged_in() {
    if (
        !isset($_SESSION['admin_logged']) ||
        $_SESSION['ip_address'] !== $_SERVER['REMOTE_ADDR'] ||
        $_SESSION['user_agent'] !== $_SERVER['HTTP_USER_AGENT']
    ) {
        return false; // Session hijack protection
    }

    return true;
}

/*
|--------------------------------------------------------------------------
| REQUIRE LOGIN
|--------------------------------------------------------------------------
*/

function require_login() {
    if (!is_logged_in()) {
        header("Location: " . BASE_URL . "/admin/login.php");
        exit;
    }
}

/*
|--------------------------------------------------------------------------
| LOGOUT
|--------------------------------------------------------------------------
*/

function logout_admin() {
    $_SESSION = [];

    if (ini_get("session.use_cookies")) {
        $params = session_get_cookie_params();
        setcookie(
            session_name(),
            "",
            time() - 42000,
            $params["path"],
            $params["domain"],
            $params["secure"],
            $params["httponly"]
        );
    }

    session_destroy();
}
