<?php
session_start();
// Security headers
header('X-Frame-Options: DENY');
header('X-Content-Type-Options: nosniff');
header('X-XSS-Protection: 1; mode=block');

require_once __DIR__ . "/../includes/auth.php";
require_login();
require_once __DIR__ . "/../includes/config.php";
require_once __DIR__ . "/../includes/functions.php";

// Check if post ID is provided
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    $_SESSION['message'] = "Invalid post ID.";
    $_SESSION['message_type'] = 'error';
    header("Location: posts.php");
    exit();
}

$post_id = (int)$_GET['id'];

// Fetch the post
$stmt = $pdo->prepare("SELECT * FROM blog_posts WHERE id = ?");
$stmt->execute([$post_id]);
$post = $stmt->fetch(PDO::FETCH_ASSOC);

// Check if post exists
if (!$post) {
    $_SESSION['message'] = "Post not found.";
    $_SESSION['message_type'] = 'error';
    header("Location: posts.php");
    exit();
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // CSRF token check (if you implement CSRF)
    // if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
    //     die('CSRF token validation failed');
    // }
    
    // Get form data
    $title = trim($_POST['title'] ?? '');
    $content = trim($_POST['content'] ?? '');
    $excerpt = trim($_POST['excerpt'] ?? '');
    $status = $_POST['status'] ?? 'draft';
    $current_image = $_POST['current_image'] ?? '';
    
    // Validate required fields
    $errors = [];
    
    if (empty($title)) {
        $errors[] = "Title is required.";
    } elseif (strlen($title) > 200) {
        $errors[] = "Title must be less than 200 characters.";
    }
    
    if (empty($content)) {
        $errors[] = "Content is required.";
    }
    
    if (strlen($excerpt) > 500) {
        $errors[] = "Excerpt must be less than 500 characters.";
    }
    
    // Handle image upload
    $featured_image = $current_image;
    
    if (isset($_FILES['featured_image']) && $_FILES['featured_image']['error'] === UPLOAD_ERR_OK) {
        $allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
        $max_size = 5 * 1024 * 1024; // 5MB
        
        $file_type = $_FILES['featured_image']['type'];
        $file_size = $_FILES['featured_image']['size'];
        $file_tmp = $_FILES['featured_image']['tmp_name'];
        $file_name = $_FILES['featured_image']['name'];
        
        // Check file type
        if (!in_array($file_type, $allowed_types)) {
            $errors[] = "Only JPG, PNG, GIF, and WebP images are allowed.";
        }
        
        // Check file size
        if ($file_size > $max_size) {
            $errors[] = "Image size must be less than 5MB.";
        }
        
        // If no errors, process the upload
        if (empty($errors)) {
            // Generate unique filename
            $file_extension = pathinfo($file_name, PATHINFO_EXTENSION);
            $new_filename = uniqid() . '_' . time() . '.' . $file_extension;
            $upload_path = __DIR__ . "/../uploads/" . $new_filename;
            
            // Create uploads directory if it doesn't exist
            if (!file_exists(__DIR__ . "/../uploads")) {
                mkdir(__DIR__ . "/../uploads", 0777, true);
            }
            
            // Move uploaded file
            if (move_uploaded_file($file_tmp, $upload_path)) {
                // Delete old image if exists and is different from new one
                if ($current_image && $current_image !== $new_filename && file_exists(__DIR__ . "/../uploads/" . $current_image)) {
                    unlink(__DIR__ . "/../uploads/" . $current_image);
                }
                $featured_image = $new_filename;
            } else {
                $errors[] = "Failed to upload image. Please try again.";
            }
        }
    } elseif (isset($_FILES['featured_image']) && $_FILES['featured_image']['error'] !== UPLOAD_ERR_NO_FILE) {
        // Handle upload errors
        $upload_errors = [
            UPLOAD_ERR_INI_SIZE => 'The uploaded file exceeds the upload_max_filesize directive in php.ini.',
            UPLOAD_ERR_FORM_SIZE => 'The uploaded file exceeds the MAX_FILE_SIZE directive that was specified in the HTML form.',
            UPLOAD_ERR_PARTIAL => 'The uploaded file was only partially uploaded.',
            UPLOAD_ERR_NO_TMP_DIR => 'Missing a temporary folder.',
            UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk.',
            UPLOAD_ERR_EXTENSION => 'A PHP extension stopped the file upload.',
        ];
        
        $error_code = $_FILES['featured_image']['error'];
        $errors[] = $upload_errors[$error_code] ?? 'Unknown upload error.';
    }
    
    // If delete image is checked
    if (isset($_POST['delete_image']) && $_POST['delete_image'] == '1' && $current_image) {
        if (file_exists(__DIR__ . "/../uploads/" . $current_image)) {
            unlink(__DIR__ . "/../uploads/" . $current_image);
        }
        $featured_image = null;
    }
    
    // If no errors, update the post
    if (empty($errors)) {
        try {
            $pdo->beginTransaction();
            
            // Generate slug from title if not provided
            $slug = $post['slug']; // Keep existing slug or generate new one if title changed
            
            // Set published_at timestamp if status changed to published
            $published_at = $post['published_at'];
            if ($status === 'published' && $post['status'] !== 'published') {
                $published_at = date('Y-m-d H:i:s');
            } elseif ($status === 'draft' && $post['status'] === 'published') {
                $published_at = null;
            }
            
            // Update post
            $update_query = "UPDATE blog_posts SET 
                            title = :title,
                            content = :content,
                            excerpt = :excerpt,
                            featured_image = :featured_image,
                            status = :status,
                            published_at = :published_at,
                            updated_at = NOW()
                            WHERE id = :id";
            
            $stmt = $pdo->prepare($update_query);
            $stmt->execute([
                ':title' => $title,
                ':content' => $content,
                ':excerpt' => $excerpt,
                ':featured_image' => $featured_image,
                ':status' => $status,
                ':published_at' => $published_at,
                ':id' => $post_id
            ]);
            
            $pdo->commit();
            
            $_SESSION['message'] = "Post updated successfully!";
            $_SESSION['message_type'] = 'success';
            header("Location: edit-post.php?id=" . $post_id);
            exit();
            
        } catch (PDOException $e) {
            $pdo->rollBack();
            $errors[] = "Database error: " . $e->getMessage();
        }
    }
    
    // If there were errors, update the post variable with submitted values for re-display
    $post['title'] = $title;
    $post['content'] = $content;
    $post['excerpt'] = $excerpt;
    $post['status'] = $status;
    $post['featured_image'] = $featured_image;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Post - Blog Admin</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif;
            background: #f5f7fa;
            color: #333;
            line-height: 1.6;
        }
        
        .admin-container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }
        
        /* Header Styles */
        .admin-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 12px;
            padding: 25px 30px;
            margin-bottom: 30px;
            color: white;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        }
        
        .admin-header h1 {
            font-size: 28px;
            font-weight: 600;
            margin-bottom: 10px;
        }
        
        .post-info {
            display: flex;
            flex-wrap: wrap;
            gap: 15px;
            font-size: 14px;
            opacity: 0.9;
        }
        
        .post-info span {
            display: flex;
            align-items: center;
            gap: 5px;
        }
        
        .header-actions {
            display: flex;
            flex-wrap: wrap;
            gap: 15px;
            margin-top: 20px;
        }
        
        .btn {
            padding: 10px 20px;
            border-radius: 8px;
            text-decoration: none;
            font-weight: 500;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            transition: all 0.3s ease;
            border: none;
            cursor: pointer;
            font-size: 14px;
        }
        
        .btn-primary {
            background: #4CAF50;
            color: white;
        }
        
        .btn-primary:hover {
            background: #45a049;
            transform: translateY(-2px);
        }
        
        .btn-secondary {
            background: #2196F3;
            color: white;
        }
        
        .btn-secondary:hover {
            background: #1976D2;
            transform: translateY(-2px);
        }
        
        .btn-light {
            background: white;
            color: #333;
            border: 1px solid #ddd;
        }
        
        .btn-light:hover {
            background: #f8f9fa;
        }
        
        .btn-danger {
            background: #f44336;
            color: white;
        }
        
        .btn-danger:hover {
            background: #d32f2f;
        }
        
        /* Form Container */
        .form-container {
            background: white;
            border-radius: 12px;
            padding: 30px;
            box-shadow: 0 5px 20px rgba(0, 0, 0, 0.08);
        }
        
        /* Error Messages */
        .errors {
            background: #f8d7da;
            border: 1px solid #f5c6cb;
            color: #721c24;
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 25px;
            list-style-type: none;
        }
        
        .errors li {
            margin: 5px 0;
            padding-left: 20px;
            position: relative;
        }
        
        .errors li:before {
            content: "⚠️";
            position: absolute;
            left: 0;
        }
        
        /* Form Groups */
        .form-group {
            margin-bottom: 25px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #444;
            font-size: 15px;
        }
        
        .form-group .required:after {
            content: " *";
            color: #f44336;
        }
        
        .form-control {
            width: 100%;
            padding: 12px 15px;
            border: 1px solid #ddd;
            border-radius: 8px;
            font-size: 15px;
            transition: border-color 0.3s ease;
            font-family: inherit;
        }
        
        .form-control:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        textarea.form-control {
            min-height: 200px;
            resize: vertical;
        }
        
        /* Character Counters */
        .char-counter {
            font-size: 12px;
            color: #666;
            margin-top: 5px;
            text-align: right;
        }
        
        /* Image Upload Section */
        .image-upload-section {
            border: 2px dashed #ddd;
            border-radius: 10px;
            padding: 30px;
            text-align: center;
            background: #fafafa;
            margin-bottom: 20px;
        }
        
        .current-image {
            margin-bottom: 25px;
        }
        
        .image-preview {
            max-width: 100%;
            height: 250px;
            object-fit: cover;
            border-radius: 8px;
            margin: 15px auto;
            display: block;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
        }
        
        .no-image {
            background: linear-gradient(45deg, #f0f0f0, #e0e0e0);
            height: 150px;
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #888;
            margin: 15px 0;
        }
        
        .file-input-container {
            position: relative;
            margin: 20px 0;
        }
        
        .file-input {
            position: absolute;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            opacity: 0;
            cursor: pointer;
        }
        
        .file-input-label {
            display: inline-block;
            padding: 12px 25px;
            background: #667eea;
            color: white;
            border-radius: 8px;
            font-weight: 500;
            cursor: pointer;
            transition: background 0.3s ease;
        }
        
        .file-input-label:hover {
            background: #5a6fd8;
        }
        
        .file-info {
            font-size: 14px;
            color: #666;
            margin-top: 10px;
        }
        
        .checkbox-group {
            display: flex;
            align-items: center;
            gap: 10px;
            margin-top: 15px;
        }
        
        .checkbox-group label {
            font-weight: normal;
            margin: 0;
            font-size: 14px;
        }
        
        /* Status Options */
        .status-options {
            display: flex;
            gap: 20px;
            flex-wrap: wrap;
        }
        
        .status-option {
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .status-option input[type="radio"] {
            width: 18px;
            height: 18px;
        }
        
        .status-badge {
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }
        
        .status-draft {
            background: #fff3cd;
            color: #856404;
        }
        
        .status-published {
            background: #d4edda;
            color: #155724;
        }
        
        /* Form Actions */
        .form-actions {
            display: flex;
            flex-wrap: wrap;
            gap: 15px;
            justify-content: flex-end;
            padding-top: 30px;
            margin-top: 30px;
            border-top: 1px solid #eee;
        }
        
        .form-actions .btn {
            min-width: 120px;
            justify-content: center;
        }
        
        /* Preview Button */
        .preview-btn {
            background: #9c27b0;
            color: white;
        }
        
        .preview-btn:hover {
            background: #8e24aa;
        }
        
        /* Tabs for Content/Excerpt */
        .editor-tabs {
            display: flex;
            border-bottom: 2px solid #eee;
            margin-bottom: 20px;
        }
        
        .editor-tab {
            padding: 12px 25px;
            background: none;
            border: none;
            border-bottom: 3px solid transparent;
            font-weight: 500;
            color: #666;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .editor-tab.active {
            color: #667eea;
            border-bottom-color: #667eea;
        }
        
        .editor-tab:hover:not(.active) {
            color: #333;
        }
        
        /* Mobile Optimizations */
        @media (max-width: 768px) {
            .admin-container {
                padding: 15px;
            }
            
            .admin-header {
                padding: 20px;
            }
            
            .admin-header h1 {
                font-size: 24px;
            }
            
            .header-actions {
                flex-direction: column;
            }
            
            .header-actions .btn {
                width: 100%;
                justify-content: center;
            }
            
            .form-container {
                padding: 20px;
            }
            
            .form-actions {
                flex-direction: column;
            }
            
            .form-actions .btn {
                width: 100%;
            }
            
            .editor-tabs {
                flex-direction: column;
                border-bottom: none;
            }
            
            .editor-tab {
                border-bottom: none;
                border-left: 3px solid transparent;
                text-align: left;
            }
            
            .editor-tab.active {
                border-left-color: #667eea;
            }
            
            .status-options {
                flex-direction: column;
                gap: 10px;
            }
            
            .image-preview {
                height: 200px;
            }
        }
        
        /* Loading Indicator */
        .loading {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid #f3f3f3;
            border-top: 3px solid #667eea;
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin-right: 10px;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Header -->
        <div class="admin-header">
            <h1>✏️ Edit Post</h1>
            <div class="post-info">
                <span>📅 Created: <?= date('F j, Y \a\t g:i A', strtotime($post['created_at'])) ?></span>
                <span>🔄 Last Updated: <?= date('F j, Y \a\t g:i A', strtotime($post['updated_at'])) ?></span>
                <span>👁 Views: <?= $post['views'] ?></span>
                <span class="status-badge status-<?= $post['status'] ?>">
                    <?= ucfirst($post['status']) ?>
                </span>
            </div>
            <div class="header-actions">
                <a href="posts.php" class="btn btn-light">
                    ← Back to Posts
                </a>
                <a href="new-post.php" class="btn btn-secondary">
                    ＋ Create New Post
                </a>
                <a href="../elementor-snippet.php?slug=<?= urlencode($post['slug']) ?>" 
                   target="_blank" 
                   class="btn preview-btn">
                    👁️ Preview Post
                </a>
            </div>
        </div>
        
        <!-- Error Messages -->
        <?php if (!empty($errors)): ?>
            <ul class="errors">
                <?php foreach ($errors as $error): ?>
                    <li><?= htmlspecialchars($error) ?></li>
                <?php endforeach; ?>
            </ul>
        <?php endif; ?>
        
        <!-- Success Message -->
        <?php if (isset($_SESSION['message'])): ?>
            <div class="message success" style="background: #d4edda; color: #155724; padding: 15px; border-radius: 8px; margin-bottom: 25px; border: 1px solid #c3e6cb;">
                <?= htmlspecialchars($_SESSION['message']) ?>
            </div>
            <?php unset($_SESSION['message']); ?>
        <?php endif; ?>
        
        <!-- Edit Form -->
        <form method="POST" action="" enctype="multipart/form-data" class="form-container" id="postForm">
            <!-- CSRF Token (if implemented)
            <input type="hidden" name="csrf_token" value="<?= $_SESSION['csrf_token'] ?? '' ?>">
            -->
            
            <input type="hidden" name="current_image" value="<?= htmlspecialchars($post['featured_image'] ?? '') ?>">
            
            <!-- Title -->
            <div class="form-group">
                <label for="title" class="required">Post Title</label>
                <input type="text" 
                       id="title" 
                       name="title" 
                       class="form-control" 
                       value="<?= htmlspecialchars($post['title']) ?>" 
                       required
                       maxlength="200"
                       placeholder="Enter post title...">
                <div class="char-counter">
                    <span id="title-counter">0</span>/200 characters
                </div>
            </div>
            
            <!-- Content Tabs -->
            <div class="editor-tabs">
                <button type="button" class="editor-tab active" data-tab="content">Content</button>
                <button type="button" class="editor-tab" data-tab="excerpt">Excerpt</button>
            </div>
            
            <!-- Content -->
            <div class="form-group tab-content" id="content-tab">
                <label for="content" class="required">Post Content</label>
                <textarea id="content" 
                          name="content" 
                          class="form-control" 
                          rows="15"
                          placeholder="Write your post content here..."><?= htmlspecialchars($post['content']) ?></textarea>
                <div class="char-counter">
                    <span id="content-counter">0</span> characters
                </div>
            </div>
            
            <!-- Excerpt -->
            <div class="form-group tab-content" id="excerpt-tab" style="display: none;">
                <label for="excerpt">Excerpt (Optional)</label>
                <textarea id="excerpt" 
                          name="excerpt" 
                          class="form-control" 
                          rows="5"
                          placeholder="Brief summary of your post (appears in post listings)..."><?= htmlspecialchars($post['excerpt'] ?? '') ?></textarea>
                <div class="char-counter">
                    <span id="excerpt-counter">0</span>/500 characters
                </div>
            </div>
            
            <!-- Featured Image -->
            <div class="form-group">
                <label>Featured Image</label>
                <div class="image-upload-section">
                    <div class="current-image">
                        <?php if ($post['featured_image']): ?>
                            <img src="../uploads/<?= htmlspecialchars($post['featured_image']) ?>" 
                                 alt="Current featured image" 
                                 class="image-preview"
                                 id="current-image-preview"
                                 onerror="this.style.display='none'; document.getElementById('no-image').style.display='flex';">
                            <div class="no-image" id="no-image" style="display: none;">
                                Image not found or deleted
                            </div>
                        <?php else: ?>
                            <div class="no-image" id="no-image">
                                No featured image set
                            </div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="file-input-container">
                        <input type="file" 
                               id="featured_image" 
                               name="featured_image" 
                               class="file-input" 
                               accept="image/*"
                               onchange="previewImage(this)">
                        <label for="featured_image" class="file-input-label">
                            📁 Choose New Image
                        </label>
                    </div>
                    
                    <div class="file-info">
                        Recommended: 1200×630 pixels, JPG, PNG, or WebP, max 5MB
                    </div>
                    
                    <?php if ($post['featured_image']): ?>
                        <div class="checkbox-group">
                            <input type="checkbox" 
                                   id="delete_image" 
                                   name="delete_image" 
                                   value="1">
                            <label for="delete_image">Remove featured image</label>
                        </div>
                    <?php endif; ?>
                </div>
                
                <!-- Image Preview for new uploads -->
                <div id="new-image-preview" style="display: none; margin-top: 20px;">
                    <h4 style="margin-bottom: 10px;">New Image Preview:</h4>
                    <img id="preview" class="image-preview" alt="New image preview">
                </div>
            </div>
            
            <!-- Status -->
            <div class="form-group">
                <label>Status</label>
                <div class="status-options">
                    <label class="status-option">
                        <input type="radio" 
                               name="status" 
                               value="draft" 
                               <?= ($post['status'] ?? 'draft') === 'draft' ? 'checked' : '' ?>>
                        <span class="status-badge status-draft">Draft</span>
                        <span>Save as draft</span>
                    </label>
                    <label class="status-option">
                        <input type="radio" 
                               name="status" 
                               value="published" 
                               <?= ($post['status'] ?? 'draft') === 'published' ? 'checked' : '' ?>>
                        <span class="status-badge status-published">Published</span>
                        <span>Publish immediately</span>
                    </label>
                </div>
            </div>
            
            <!-- Form Actions -->
            <div class="form-actions">
                <a href="posts.php" class="btn btn-light">Cancel</a>
                <button type="submit" name="save_draft" value="1" class="btn btn-light">
                    💾 Save as Draft
                </button>
                <button type="submit" name="publish" value="1" class="btn btn-primary">
                    🚀 Update & Publish
                </button>
            </div>
        </form>
    </div>
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Character counters
            const titleInput = document.getElementById('title');
            const contentInput = document.getElementById('content');
            const excerptInput = document.getElementById('excerpt');
            
            const titleCounter = document.getElementById('title-counter');
            const contentCounter = document.getElementById('content-counter');
            const excerptCounter = document.getElementById('excerpt-counter');
            
            function updateCounters() {
                titleCounter.textContent = titleInput.value.length;
                contentCounter.textContent = contentInput.value.length;
                excerptCounter.textContent = excerptInput.value.length;
            }
            
            [titleInput, contentInput, excerptInput].forEach(input => {
                input.addEventListener('input', updateCounters);
            });
            
            updateCounters(); // Initial count
            
            // Tab functionality
            const tabs = document.querySelectorAll('.editor-tab');
            const tabContents = document.querySelectorAll('.tab-content');
            
            tabs.forEach(tab => {
                tab.addEventListener('click', function() {
                    const tabId = this.getAttribute('data-tab');
                    
                    // Update active tab
                    tabs.forEach(t => t.classList.remove('active'));
                    this.classList.add('active');
                    
                    // Show active content
                    tabContents.forEach(content => {
                        if (content.id === tabId + '-tab') {
                            content.style.display = 'block';
                        } else {
                            content.style.display = 'none';
                        }
                    });
                });
            });
            
            // Auto-set status based on button clicked
            const form = document.getElementById('postForm');
            const statusInputs = document.querySelectorAll('input[name="status"]');
            
            form.addEventListener('submit', function(e) {
                const submitButton = e.submitter || e.explicitOriginalTarget;
                
                if (submitButton && submitButton.name === 'save_draft') {
                    // Ensure draft status is selected
                    statusInputs[0].checked = true;
                } else if (submitButton && submitButton.name === 'publish') {
                    // Ensure published status is selected
                    statusInputs[1].checked = true;
                }
                
                // Show loading indicator
                if (submitButton) {
                    const originalText = submitButton.innerHTML;
                    submitButton.innerHTML = '<span class="loading"></span> Updating...';
                    submitButton.disabled = true;
                    
                    // Re-enable button if form doesn't submit (due to errors)
                    setTimeout(() => {
                        submitButton.innerHTML = originalText;
                        submitButton.disabled = false;
                    }, 5000);
                }
            });
            
            // Auto-save reminder
            let saveTimer;
            let isDirty = false;
            
            [titleInput, contentInput, excerptInput].forEach(input => {
                input.addEventListener('input', function() {
                    isDirty = true;
                    clearTimeout(saveTimer);
                    
                    // Show save reminder
                    const saveReminder = document.getElementById('save-reminder');
                    if (saveReminder) {
                        saveReminder.style.display = 'block';
                        setTimeout(() => {
                            saveReminder.style.display = 'none';
                        }, 3000);
                    }
                });
            });
            
            // Warn before leaving unsaved changes
            window.addEventListener('beforeunload', function(e) {
                if (isDirty) {
                    e.preventDefault();
                    e.returnValue = 'You have unsaved changes. Are you sure you want to leave?';
                }
            });
            
            // Form reset clears dirty flag
            form.addEventListener('submit', function() {
                isDirty = false;
            });
        });
        
        // Image preview for new uploads
        function previewImage(input) {
            const preview = document.getElementById('preview');
            const previewContainer = document.getElementById('new-image-preview');
            const currentPreview = document.getElementById('current-image-preview');
            const noImage = document.getElementById('no-image');
            
            if (input.files && input.files[0]) {
                const reader = new FileReader();
                
                reader.onload = function(e) {
                    preview.src = e.target.result;
                    previewContainer.style.display = 'block';
                    
                    // Hide current image preview
                    if (currentPreview) currentPreview.style.display = 'none';
                    if (noImage) noImage.style.display = 'none';
                }
                
                reader.readAsDataURL(input.files[0]);
            } else {
                previewContainer.style.display = 'none';
            }
        }
        
        // Auto-generate excerpt from first paragraph if empty
        function generateExcerpt() {
            const content = document.getElementById('content').value;
            const excerptInput = document.getElementById('excerpt');
            
            if (!excerptInput.value.trim() && content.trim()) {
                // Get first paragraph or first 150 characters
                const firstParagraph = content.split('\n\n')[0] || content;
                const excerpt = firstParagraph.substring(0, 150).trim();
                if (excerpt.length === 150) excerpt += '...';
                
                excerptInput.value = excerpt;
                document.getElementById('excerpt-counter').textContent = excerpt.length;
            }
        }
        
        // Optional: Add a button to generate excerpt
        const excerptTab = document.querySelector('[data-tab="excerpt"]');
        excerptTab.addEventListener('click', function() {
            if (!document.getElementById('excerpt').value.trim()) {
                generateExcerpt();
            }
        });
    </script>
</body>
</html>